<?php
session_start();
require_once '../config/database.php';

header('Content-Type: application/json');

// 1. Check Admin Auth
if (!isset($_SESSION['admin_id'])) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit;
}

// 2. Get JSON Input
$input = json_decode(file_get_contents('php://input'), true);

$unique_id = trim($input['unique_id'] ?? '');
$game_id = filter_var($input['game_id'] ?? 0, FILTER_VALIDATE_INT);
$points_to_add = filter_var($input['points'] ?? 0, FILTER_VALIDATE_INT);
$note = trim($input['note'] ?? '');

// 3. Validation
if (empty($unique_id)) {
    echo json_encode(['success' => false, 'message' => 'Participant ID is required']);
    exit;
}
if (!$game_id) {
    echo json_encode(['success' => false, 'message' => 'Game ID is required']);
    exit;
}
if ($points_to_add <= 0) {
    echo json_encode(['success' => false, 'message' => 'Points must be a positive number']);
    exit;
}

// 3.5 Check Permissions (If not Super Admin, must be assigned to this game)
if (($_SESSION['role'] ?? '') !== 'super') {
    $stmt = $pdo->prepare("SELECT game_id FROM admins WHERE id = ?");
    $stmt->execute([$_SESSION['admin_id']]);
    $assignedGameId = $stmt->fetchColumn();

    if ($assignedGameId != $game_id) {
        echo json_encode(['success' => false, 'message' => 'Unauthorized: You can only assign points for your assigned game.']);
        exit;
    }
}

try {
    // 8. Use transactions
    $pdo->beginTransaction();

    // Check participant exists
    $stmt = $pdo->prepare("SELECT id, name FROM participants WHERE unique_id = ?");
    $stmt->execute([$unique_id]);
    $participant = $stmt->fetch();

    if (!$participant) {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => 'Participant not found']);
        exit;
    }
    $participant_id = $participant['id'];

    // Check game exists and get max_points
    $stmt = $pdo->prepare("SELECT id, max_points FROM games WHERE id = ?");
    $stmt->execute([$game_id]);
    $game = $stmt->fetch();

    if (!$game) {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => 'Game not found']);
        exit;
    }
    $max_points = (int)$game['max_points'];

    // 4. Max score logic
    // Fetch current points (Locking row if exists)
    $stmt = $pdo->prepare("SELECT id, points FROM participant_points WHERE participant_id = ? AND game_id = ? FOR UPDATE");
    $stmt->execute([$participant_id, $game_id]);
    $current_entry = $stmt->fetch();

    $current_points = $current_entry ? (int)$current_entry['points'] : 0;

    // Calculate new score
    $proposed_total = $current_points + $points_to_add;
    $new_score = min($proposed_total, $max_points);
    $actual_added = $new_score - $current_points;

    if ($actual_added === 0) {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => 'Participant already reached maximum points']);
        exit;
    }

    // 5. Update participant_points table
    if ($current_entry) {
        $updateStmt = $pdo->prepare("UPDATE participant_points SET points = ? WHERE id = ?");
        $updateStmt->execute([$new_score, $current_entry['id']]);
    } else {
        $insertStmt = $pdo->prepare("INSERT INTO participant_points (participant_id, game_id, points) VALUES (?, ?, ?)");
        $insertStmt->execute([$participant_id, $game_id, $new_score]);
    }

    // 6. Insert admin action log
    $stmt = $pdo->prepare("INSERT INTO admin_logs (admin_id, participant_id, game_id, points_given, note, ip) VALUES (?, ?, ?, ?, ?, ?)");
    $stmt->execute([$_SESSION['admin_id'], $participant_id, $game_id, $actual_added, $note, $_SERVER['REMOTE_ADDR']]);
    $log_id = $pdo->lastInsertId();

    $pdo->commit();

    // 7. Return JSON
    echo json_encode([
        'success' => true,
        'message' => 'Points updated successfully',
        'added' => $actual_added,
        'new_points' => $new_score,
        'max_points' => $max_points,
        'log_id' => $log_id
    ]);

} catch (Exception $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    echo json_encode(['success' => false, 'message' => 'Database Error: ' . $e->getMessage()]);
}
?>