<?php
$GLOBALS['path_depth'] = 1;
require_once '../../config/database.php';

// Ensure user is logged in and is a regular admin (game controller)
if (!isset($_SESSION['admin_id']) || ($_SESSION['role'] ?? '') !== 'admin') {
    header("Location: ../game_login.php");
    exit;
}

// Fetch Admin and Assigned Game Details
$stmt = $pdo->prepare("
    SELECT a.username, g.id as game_id, g.game_name, g.max_points 
    FROM admins a 
    LEFT JOIN games g ON a.game_id = g.id 
    WHERE a.id = ?
");
$stmt->execute([$_SESSION['admin_id']]);
$adminData = $stmt->fetch();

if (!$adminData || !$adminData['game_id']) {
    $error = "No game assigned to this controller account. Please contact Super Admin.";
}

// Fetch Recent Logs for this Admin
$logStmt = $pdo->prepare("
    SELECT l.*, p.name as participant_name, p.unique_id 
    FROM admin_logs l
    JOIN participants p ON l.participant_id = p.id
    WHERE l.admin_id = ?
    ORDER BY l.created_at DESC
    LIMIT 10
");
$logStmt->execute([$_SESSION['admin_id']]);
$recentLogs = $logStmt->fetchAll();
?>
<!DOCTYPE html>
<html class="dark" lang="en">
<head>
    <meta charset="utf-8"/>
    <meta content="width=device-width, initial-scale=1.0" name="viewport"/>
    <title>Game Controller Dashboard</title>
    <link href="https://fonts.googleapis.com/css2?family=Spline+Sans:wght@300;400;500;600;700&display=swap" rel="stylesheet"/>
    <link href="https://fonts.googleapis.com/css2?family=Material+Symbols+Outlined:wght,FILL@100..700,0..1&display=swap" rel="stylesheet"/>
    <script src="https://cdn.tailwindcss.com?plugins=forms,container-queries"></script>
    <script>
        tailwind.config = {
            darkMode: "class",
            theme: {
                extend: {
                    colors: {
                        "primary": "#f9f506",
                        "background-dark": "#1a1a1a",
                        "surface-dark": "#232310",
                    },
                    fontFamily: {
                        "display": ["Spline Sans", "sans-serif"]
                    }
                },
            },
        }
    </script>
    <style>
        body { background-color: #1a1a1a; color: white; font-family: 'Spline Sans', sans-serif; }
        .glass-panel { background: rgba(35, 35, 16, 0.6); backdrop-filter: blur(12px); border: 1px solid rgba(255, 255, 255, 0.1); }
    </style>
</head>
<body class="min-h-screen flex flex-col">
    
    <!-- Header -->
    <header class="h-16 border-b border-white/10 flex items-center justify-between px-6 bg-surface-dark/50 backdrop-blur-md sticky top-0 z-50">
        <div class="flex items-center gap-3">
            <span class="material-symbols-outlined text-primary">sports_esports</span>
            <h1 class="font-bold tracking-wider text-lg">MAGMA <span class="text-primary text-xs align-top">CONTROLLER</span></h1>
        </div>
        <div class="flex items-center gap-4">
            <span class="text-sm text-gray-400 hidden md:inline"><?php echo htmlspecialchars($adminData['username']); ?></span>
            <a href="logout.php" class="text-xs font-bold text-red-400 hover:text-red-300 uppercase tracking-widest">Logout</a>
        </div>
    </header>

    <main class="flex-1 p-4 md:p-8 max-w-4xl mx-auto w-full">
        <?php if (isset($error)): ?>
            <div class="p-4 mb-6 bg-red-900/20 border border-red-500/50 rounded-xl text-red-200 text-center">
                <?php echo $error; ?>
            </div>
        <?php else: ?>
            
            <!-- Game Info Card -->
            <div class="mb-8 text-center">
                <h2 class="text-3xl font-bold text-white mb-2"><?php echo htmlspecialchars($adminData['game_name']); ?></h2>
                <div class="inline-flex items-center gap-2 px-3 py-1 rounded-full bg-primary/10 border border-primary/20 text-primary text-xs font-bold uppercase tracking-wider">
                    <span class="material-symbols-outlined text-sm">stars</span>
                    Max Points: <?php echo $adminData['max_points']; ?>
                </div>
            </div>

            <!-- Point Assignment Form -->
            <div class="glass-panel rounded-2xl p-6 md:p-8 shadow-2xl mb-8">
                <form id="assignPointsForm" class="space-y-6">
                    <input type="hidden" name="game_id" value="<?php echo $adminData['game_id']; ?>">
                    
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div class="space-y-2">
                            <label class="text-xs font-bold text-gray-400 uppercase tracking-wider">Participant ID</label>
                            <div class="relative">
                                <div class="absolute inset-y-0 left-0 pl-4 flex items-center pointer-events-none">
                                    <span class="material-symbols-outlined text-gray-500">qr_code_scanner</span>
                                </div>
                                <input type="text" name="unique_id" id="unique_id" required placeholder="e.g. A1B2" 
                                    class="w-full bg-black/40 border border-white/10 rounded-xl py-3 pl-12 pr-4 text-white placeholder-gray-600 focus:border-primary focus:ring-1 focus:ring-primary outline-none transition-all font-mono uppercase text-lg">
                            </div>
                        </div>
                        
                        <div class="space-y-2">
                            <label class="text-xs font-bold text-gray-400 uppercase tracking-wider">Points to Add</label>
                            <div class="relative">
                                <div class="absolute inset-y-0 left-0 pl-4 flex items-center pointer-events-none">
                                    <span class="material-symbols-outlined text-gray-500">add_circle</span>
                                </div>
                                <input type="number" name="points" required min="1" max="<?php echo $adminData['max_points']; ?>" placeholder="0" 
                                    class="w-full bg-black/40 border border-white/10 rounded-xl py-3 pl-12 pr-4 text-white placeholder-gray-600 focus:border-primary focus:ring-1 focus:ring-primary outline-none transition-all font-mono text-lg">
                            </div>
                        </div>
                    </div>

                    <button type="submit" id="submitBtn" class="w-full bg-primary hover:bg-yellow-400 text-black font-bold py-4 rounded-xl uppercase tracking-widest shadow-lg shadow-primary/20 transition-all active:scale-[0.98] flex items-center justify-center gap-2">
                        <span>Assign Points</span>
                        <span class="material-symbols-outlined">send</span>
                    </button>
                </form>
                <div id="responseMessage" class="mt-4 hidden p-4 rounded-xl text-center text-sm font-bold"></div>
            </div>

            <!-- Recent Activity -->
            <div>
                <h3 class="text-sm font-bold text-gray-500 uppercase tracking-wider mb-4 px-2">Recent Activity</h3>
                <div class="glass-panel rounded-xl overflow-hidden">
                    <div class="overflow-x-auto">
                        <table class="w-full text-left text-sm">
                            <thead class="bg-white/5 text-gray-400 border-b border-white/5">
                                <tr>
                                    <th class="p-4 font-medium">Time</th>
                                    <th class="p-4 font-medium">Participant</th>
                                    <th class="p-4 font-medium text-right">Points</th>
                                </tr>
                            </thead>
                            <tbody class="divide-y divide-white/5" id="activityLog">
                                <?php foreach ($recentLogs as $log): ?>
                                <tr class="hover:bg-white/5 transition-colors">
                                    <td class="p-4 text-gray-500 whitespace-nowrap"><?php echo date('H:i:s', strtotime($log['created_at'])); ?></td>
                                    <td class="p-4">
                                        <div class="font-bold text-white"><?php echo htmlspecialchars($log['participant_name']); ?></div>
                                        <div class="text-xs text-gray-500 font-mono"><?php echo htmlspecialchars($log['unique_id']); ?></div>
                                    </td>
                                    <td class="p-4 text-right font-mono text-primary">+<?php echo $log['points_given']; ?></td>
                                </tr>
                                <?php endforeach; ?>
                                <?php if (empty($recentLogs)): ?>
                                    <tr><td colspan="3" class="p-6 text-center text-gray-500">No points assigned yet today.</td></tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

        <?php endif; ?>
    </main>

    <script>
        document.getElementById('assignPointsForm').addEventListener('submit', async function(e) {
            e.preventDefault();
            const btn = document.getElementById('submitBtn');
            const msgDiv = document.getElementById('responseMessage');
            const formData = new FormData(this);
            const data = Object.fromEntries(formData.entries());

            // UI Loading State
            btn.disabled = true;
            btn.innerHTML = '<span class="material-symbols-outlined animate-spin">progress_activity</span> Processing...';
            msgDiv.classList.add('hidden');
            msgDiv.className = 'mt-4 hidden p-4 rounded-xl text-center text-sm font-bold'; // Reset classes

            try {
                const response = await fetch('../assign_points.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify(data)
                });
                
                const result = await response.json();

                msgDiv.classList.remove('hidden');
                if (result.success) {
                    msgDiv.classList.add('bg-green-500/20', 'text-green-400', 'border', 'border-green-500/30');
                    msgDiv.innerHTML = `<span class="block text-lg">✅ Success</span>${result.message}. Total: ${result.new_points}`;
                    document.getElementById('unique_id').value = ''; // Clear ID only
                    document.getElementById('unique_id').focus();
                    
                    // Optional: Reload page to update log or append row via JS (Reload is simpler for now)
                    setTimeout(() => location.reload(), 1500);
                } else {
                    msgDiv.classList.add('bg-red-500/20', 'text-red-400', 'border', 'border-red-500/30');
                    msgDiv.innerHTML = `❌ ${result.message}`;
                }
            } catch (error) {
                msgDiv.classList.remove('hidden');
                msgDiv.classList.add('bg-red-500/20', 'text-red-400');
                msgDiv.innerText = '❌ Network or Server Error';
            } finally {
                btn.disabled = false;
                btn.innerHTML = '<span>Assign Points</span><span class="material-symbols-outlined">send</span>';
            }
        });
    </script>
</body>
</html>