<?php
$GLOBALS['path_depth'] = 1;
require_once '../../config/database.php';

// Ensure user is logged in and is a super_admin
if (!isset($_SESSION['admin_id']) || ($_SESSION['role'] ?? '') !== 'super') {
    header("Location: ../admin_login.php");
    exit;
}

$logs = [];
$error = '';

try {
    // Attempt to fetch logs. Assumes a system_logs table exists.
    // If you haven't created this table yet, run:
    // CREATE TABLE system_logs (id INT AUTO_INCREMENT PRIMARY KEY, admin_id INT, action VARCHAR(50), details TEXT, ip_address VARCHAR(45), created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP);
    $sql = "SELECT l.*, a.username, p.name as participant_name, g.game_name 
            FROM admin_logs l 
            LEFT JOIN admins a ON l.admin_id = a.id 
            LEFT JOIN participants p ON l.participant_id = p.id
            LEFT JOIN games g ON l.game_id = g.id
            ORDER BY l.created_at DESC 
            LIMIT 100";
    $logs = $pdo->query($sql)->fetchAll();
} catch (PDOException $e) {
    // Gracefully handle missing table
    $error = "Logs are currently unavailable. Error: " . $e->getMessage();
}
?>
<!DOCTYPE html>
<html class="dark" lang="en">
<head>
    <meta charset="utf-8"/>
    <meta content="width=device-width, initial-scale=1.0" name="viewport"/>
    <title>MAGMA System Logs</title>
    <link href="https://fonts.googleapis.com/css2?family=Spline+Sans:wght@300;400;500;600;700&display=swap" rel="stylesheet"/>
    <link href="https://fonts.googleapis.com/css2?family=Material+Symbols+Outlined:wght,FILL@100..700,0..1&display=swap" rel="stylesheet"/>
    <script src="https://cdn.tailwindcss.com?plugins=forms,container-queries"></script>
    <script>
        tailwind.config = {
            darkMode: "class",
            theme: {
                extend: {
                    colors: {
                        "primary": "#ff1a1a",
                        "background-dark": "#0a0a0a",
                        "surface-dark": "#121212",
                    },
                    fontFamily: {
                        "display": ["Spline Sans", "sans-serif"]
                    }
                },
            },
        }
    </script>
    <link rel="stylesheet" href="../css/admin_dashboard.css">
</head>
<body class="bg-background-dark text-white font-display flex h-screen overflow-hidden">
    
    <!-- Sidebar -->
    <aside class="w-64 bg-surface-dark border-r border-white/5 flex flex-col z-20 hidden md:flex">
        <div class="p-6 flex items-center gap-3 border-b border-white/5">
            <div class="w-10 h-10 rounded-lg bg-primary/20 flex items-center justify-center border border-primary/30">
                <span class="material-symbols-outlined text-primary">admin_panel_settings</span>
            </div>
            <div>
                <h1 class="font-bold text-lg tracking-wider">MAGMA</h1>
                <p class="text-[10px] text-gray-500 uppercase tracking-widest font-bold">Super Admin</p>
            </div>
        </div>
        
        <nav class="flex-1 py-6 space-y-1">
            <a href="dashboard.php" class="sidebar-link flex items-center gap-3 px-6 py-3 text-sm font-medium text-gray-300">
                <span class="material-symbols-outlined">dashboard</span>
                Dashboard
            </a>
            <a href="controllers.php" class="sidebar-link flex items-center gap-3 px-6 py-3 text-sm font-medium text-gray-300">
                <span class="material-symbols-outlined">sports_esports</span>
                Manage Controllers
            </a>
            <a href="games.php" class="sidebar-link flex items-center gap-3 px-6 py-3 text-sm font-medium text-gray-300">
                <span class="material-symbols-outlined">videogame_asset</span>
                Games & Scoring
            </a>
            <a href="participants.php" class="sidebar-link flex items-center gap-3 px-6 py-3 text-sm font-medium text-gray-300">
                <span class="material-symbols-outlined">groups</span>
                Participants
            </a>
            <a href="requests.php" class="sidebar-link flex items-center gap-3 px-6 py-3 text-sm font-medium text-gray-300">
                <span class="material-symbols-outlined">pending_actions</span>
                Requests
            </a>
            <a href="logs.php" class="sidebar-link active flex items-center gap-3 px-6 py-3 text-sm font-medium text-gray-300">
                <span class="material-symbols-outlined">receipt_long</span>
                System Logs
            </a>
        </nav>

        <div class="p-4 border-t border-white/5">
            <a href="logout.php" class="flex items-center gap-3 px-4 py-3 text-sm font-medium text-red-400 hover:bg-red-500/10 rounded-lg transition-colors">
                <span class="material-symbols-outlined">logout</span>
                Sign Out
            </a>
        </div>
    </aside>

    <!-- Main Content -->
    <main class="flex-1 flex flex-col relative overflow-hidden">
        <div class="absolute inset-0 geo-overlay z-0"></div>
        
        <!-- Mobile Header -->
        <header class="md:hidden h-16 border-b border-white/5 flex items-center justify-between px-4 bg-surface-dark z-20">
            <div class="flex items-center gap-2">
                <span class="material-symbols-outlined text-primary">admin_panel_settings</span>
                <span class="font-bold tracking-wider">MAGMA</span>
            </div>
            <button class="text-gray-400 hover:text-white">
                <span class="material-symbols-outlined">menu</span>
            </button>
        </header>

        <div class="flex-1 overflow-y-auto p-6 md:p-10 relative z-10 custom-scrollbar">
            <div class="max-w-7xl mx-auto">
                <div class="flex items-end justify-between mb-8">
                    <div>
                        <h2 class="text-3xl font-bold text-white mb-1">System Logs</h2>
                        <p class="text-gray-500 text-sm">View recent system activity and audit trails.</p>
                    </div>
                </div>

                <?php if ($error): ?>
                    <div class="p-4 mb-6 text-sm text-yellow-400 bg-yellow-900/20 border border-yellow-500/30 rounded-xl flex items-center gap-2">
                        <span class="material-symbols-outlined">warning</span>
                        <?php echo htmlspecialchars($error); ?>
                    </div>
                <?php endif; ?>

                <!-- Logs Table -->
                <div class="glass-panel rounded-2xl overflow-hidden">
                    <div class="overflow-x-auto">
                        <table class="w-full text-left border-collapse">
                            <thead>
                                <tr class="bg-white/5 border-b border-white/5">
                                    <th class="p-4 pl-6 text-xs font-bold text-gray-500 uppercase tracking-wider">Time</th>
                                    <th class="p-4 text-xs font-bold text-gray-500 uppercase tracking-wider">User</th>
                                    <th class="p-4 text-xs font-bold text-gray-500 uppercase tracking-wider">Activity</th>
                                    <th class="p-4 text-xs font-bold text-gray-500 uppercase tracking-wider">Note</th>
                                    <th class="p-4 text-xs font-bold text-gray-500 uppercase tracking-wider text-right pr-6">IP</th>
                                </tr>
                            </thead>
                            <tbody class="divide-y divide-white/5 font-mono text-sm">
                                <?php foreach ($logs as $log): ?>
                                <tr class="hover:bg-white/5 transition-colors">
                                    <td class="p-4 pl-6 text-gray-400 whitespace-nowrap"><?php echo htmlspecialchars($log['created_at']); ?></td>
                                    <td class="p-4 text-primary font-bold"><?php echo htmlspecialchars($log['username'] ?? 'System'); ?></td>
                                    <td class="p-4 text-gray-300">
                                        Assigned <span class="text-white font-bold"><?php echo htmlspecialchars($log['points_given']); ?> pts</span>
                                        to <span class="text-white"><?php echo htmlspecialchars($log['participant_name']); ?></span>
                                        for <span class="text-gray-400"><?php echo htmlspecialchars($log['game_name']); ?></span>
                                    </td>
                                    <td class="p-4 text-gray-500 italic"><?php echo htmlspecialchars($log['note'] ?? '-'); ?></td>
                                    <td class="p-4 pr-6 text-right text-gray-500"><?php echo htmlspecialchars($log['ip'] ?? ''); ?></td>
                                </tr>
                                <?php endforeach; ?>
                                <?php if (empty($logs) && !$error): ?>
                                    <tr>
                                        <td colspan="5" class="p-8 text-center text-gray-500 font-sans">No logs found.</td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </main>
</body>
</html>