<?php
require_once '../config/database.php';

// Access Control
if (!isset($_SESSION['admin_id'])) {
    header("Location: admin_login.php");
    exit;
}
if ($_SESSION['role'] !== 'super') {
    die('<div class="container mt-5"><div class="alert alert-danger">Access Denied. Super Admin only. <a href="admin_dashboard.php">Back</a></div></div>');
}

$search = trim($_GET['search'] ?? '');
$page = filter_input(INPUT_GET, 'page', FILTER_VALIDATE_INT) ?: 1;
$limit = 10;
$offset = ($page - 1) * $limit;

$params = [];

// Build Query
$sql = "SELECT 
            al.id,
            a.username AS admin_name,
            p.name AS participant_name,
            g.game_name,
            al.points_given,
            al.note,
            al.ip,
            al.created_at
        FROM admin_logs al
        JOIN admins a ON al.admin_id = a.id
        JOIN participants p ON al.participant_id = p.id
        JOIN games g ON al.game_id = g.id";

if (!empty($search)) {
    $sql .= " WHERE p.name LIKE ? OR a.username LIKE ? OR g.game_name LIKE ?";
    $searchTerm = "%$search%";
    $params = [$searchTerm, $searchTerm, $searchTerm];
}

// Count Total for Pagination
$countSql = "SELECT COUNT(*) FROM (" . $sql . ") as count_table";
$countStmt = $pdo->prepare($countSql);
$countStmt->execute($params);
$totalLogs = $countStmt->fetchColumn();
$totalPages = ceil($totalLogs / $limit);

// Fetch Data
$sql .= " ORDER BY al.created_at DESC LIMIT $limit OFFSET $offset";

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$logs = $stmt->fetchAll();

include '../views/header.php';
?>
<!-- Page CSS -->
<link rel="stylesheet" href="css/admin_logs.css">

<div>
    <div class="d-flex justify-content-between align-items-center mb-4 flex-wrap gap-2">
        <h2 class="mb-0">Activity Logs</h2>
        <form class="d-flex" method="GET" action="admin_logs.php">
            <input class="form-control me-2" type="search" name="search" placeholder="Search logs..." value="<?php echo htmlspecialchars($search); ?>">
            <button class="btn btn-outline-primary" type="submit">Search</button>
            <?php if($search): ?>
                <a href="admin_logs.php" class="btn btn-outline-secondary ms-2">Reset</a>
            <?php endif; ?>
        </form>
    </div>

    <div class="logs-table-container shadow-soft">
        <div class="table-responsive">
            <table class="table table-hover align-middle">
                <thead class="table-light">
                    <tr>
                        <th>Time</th>
                        <th>Admin</th>
                        <th>Participant</th>
                        <th>Game</th>
                        <th>Points</th>
                        <th>Note</th>
                        <th>IP</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($logs as $log): ?>
                    <tr>
                        <td class="text-muted small text-nowrap"><?php echo date('M d, H:i', strtotime($log['created_at'])); ?></td>
                        <td class="fw-bold text-primary"><?php echo htmlspecialchars($log['admin_name']); ?></td>
                        <td><?php echo htmlspecialchars($log['participant_name']); ?></td>
                        <td><?php echo htmlspecialchars($log['game_name']); ?></td>
                        <td>
                            <span class="badge bg-success badge-points">+<?php echo $log['points_given']; ?></span>
                        </td>
                        <td class="text-muted small"><?php echo htmlspecialchars($log['note']); ?></td>
                        <td class="text-muted small"><?php echo htmlspecialchars($log['ip']); ?></td>
                    </tr>
                    <?php endforeach; ?>
                    <?php if (empty($logs)): ?>
                        <tr><td colspan="7" class="text-center text-muted py-4">No logs found.</td></tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>

    <!-- Pagination -->
    <?php if ($totalPages > 1): ?>
    <nav class="mt-4">
        <ul class="pagination justify-content-center">
            <?php for ($i = 1; $i <= $totalPages; $i++): ?>
                <li class="page-item <?php echo ($i == $page) ? 'active' : ''; ?>">
                    <a class="page-link" href="?page=<?php echo $i; ?>&search=<?php echo urlencode($search); ?>"><?php echo $i; ?></a>
                </li>
            <?php endfor; ?>
        </ul>
    </nav>
    <?php endif; ?>
</div>

<?php include '../views/footer.php'; ?>