<?php
// public/profile.php
require_once '../config/database.php';

if (!isset($_SESSION['unique_id'])) {
    header("Location: login.php");
    exit;
}

$unique_id = $_SESSION['unique_id'];

// Fetch Participant Details
$stmt = $pdo->prepare("SELECT * FROM participants WHERE unique_id = ?");
$stmt->execute([$unique_id]);
$user = $stmt->fetch();

if (!$user) {
    session_destroy();
    header("Location: register.php");
    exit;
}

// Fetch Game Scores (Left Join to show all games even if 0 points)
$sql = "SELECT g.game_name, g.max_points, COALESCE(pp.points, 0) as points
        FROM games g
        LEFT JOIN participant_points pp ON g.id = pp.game_id AND pp.participant_id = ?
        ORDER BY g.game_name ASC";
$stmt = $pdo->prepare($sql);
$stmt->execute([$user['id']]);
$gameScores = $stmt->fetchAll();

// Calculate Total Points
$totalPoints = 0;
foreach ($gameScores as $score) {
    $totalPoints += $score['points'];
}

// Check for pending edit requests
$reqStmt = $pdo->prepare("SELECT id FROM profile_edit_requests WHERE participant_id = ? AND status = 'pending'");
$reqStmt->execute([$user['id']]);
$hasPendingRequest = $reqStmt->fetch();

// Generate QR Code URL
$qrUrl = "https://api.qrserver.com/v1/create-qr-code/?size=300x300&data=" . urlencode($user['unique_id']);

include '../views/header.php';
?>
<!-- Link CSS -->
<link rel="stylesheet" href="css/profile.css">

<div class="row g-4">
    <!-- Left Column: Profile & QR -->
    <div class="col-md-4">
        <div class="card shadow-soft profile-card h-100">
            <div class="card-body text-center">
                <div class="qr-container mb-3 border">
                    <img src="<?php echo $qrUrl; ?>" alt="QR Code" class="img-fluid" width="180">
                </div>
                
                <h4 class="card-title fw-bold mb-1"><?php echo htmlspecialchars($user['name']); ?></h4>
                <div class="unique-id-box mb-3 text-primary"><?php echo htmlspecialchars($user['unique_id']); ?></div>
                
                <ul class="list-group list-group-flush text-start">
                    <li class="list-group-item">
                        <small class="text-muted">College</small>
                        <div class="fw-medium"><?php echo htmlspecialchars($user['college']); ?></div>
                    </li>
                    <li class="list-group-item">
                        <small class="text-muted">Department</small>
                        <div class="fw-medium"><?php echo htmlspecialchars($user['department']); ?></div>
                    </li>
                    <li class="list-group-item">
                        <small class="text-muted">Year</small>
                        <div class="fw-medium"><?php echo htmlspecialchars($user['year']); ?> Year</div>
                    </li>
                    <li class="list-group-item">
                        <small class="text-muted">Phone</small>
                        <div class="fw-medium"><?php echo htmlspecialchars($user['phone']); ?></div>
                    </li>
                </ul>
            </div>
            <div class="card-footer text-muted text-center small">
                Registered: <?php echo date('M d, Y', strtotime($user['created_at'])); ?>
            </div>
            <!-- Edit Profile Button -->
            <div class="card-body border-top no-print">
                <?php if ($hasPendingRequest): ?>
                    <button class="btn btn-warning w-100 mb-2" disabled>Edit Pending Approval</button>
                <?php else: ?>
                    <a href="edit_profile.php" class="btn btn-outline-secondary w-100 mb-2">Edit Profile</a>
                <?php endif; ?>
            </div>
            <!-- Download/Print Button -->
            <div class="card-body border-top no-print">
                <button onclick="window.print()" class="btn btn-outline-primary w-100">Download QR Card (PDF)</button>
            </div>
        </div>
    </div>

    <!-- Right Column: Stats & Scores -->
    <div class="col-md-8">
        <!-- Total Points Card -->
        <div class="card shadow-soft total-points-card mb-4">
            <div class="card-body d-flex justify-content-between align-items-center p-4">
                <div>
                    <h5 class="mb-0 opacity-75">Total Score</h5>
                    <h1 class="display-4 fw-bold mb-0"><?php echo $totalPoints; ?></h1>
                </div>
                <div style="font-size: 3rem; opacity: 0.5;">🏆</div>
            </div>
        </div>

        <!-- Game Scores Table -->
        <div class="card shadow-soft">
            <div class="card-header bg-white py-3">
                <h5 class="mb-0 fw-bold">Game Performance</h5>
            </div>
            <div class="card-body p-0">
                <?php if (count($gameScores) > 0): ?>
                    <div class="table-responsive">
                        <table class="table table-hover mb-0 align-middle">
                            <thead class="table-light">
                                <tr>
                                    <th class="ps-4">Game Name</th>
                                    <th class="text-center">Max Points</th>
                                    <th class="text-center pe-4">Score</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($gameScores as $game): ?>
                                    <tr>
                                        <td class="ps-4 fw-medium"><?php echo htmlspecialchars($game['game_name']); ?></td>
                                        <td class="text-center text-muted"><?php echo $game['max_points']; ?></td>
                                        <td class="text-center pe-4">
                                            <?php if ($game['points'] > 0): ?>
                                                <span class="badge bg-success fs-6"><?php echo $game['points']; ?></span>
                                            <?php else: ?>
                                                <span class="badge bg-secondary opacity-50">0</span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <div class="p-4 text-center text-muted">
                        No games available yet.
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<?php include '../views/footer.php'; ?>